"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WastageController = void 0;
const entities_1 = require("../entities");
const factory_1 = require("../entities/factory");
const FactoryVolatileProduct_1 = require("../entities/FactoryVolatileProduct");
const volatile_product_1 = require("../entities/volatile-product");
const wastage_1 = require("../entities/wastage");
const wastage_product_1 = require("../entities/wastage-product");
const utils_1 = require("../utils");
const base_1 = require("./base");
const typeorm_1 = require("typeorm");
class WastageController extends base_1.Controller {
    constructor() {
        super(wastage_1.Wastage);
    }
    get = (0, utils_1.requestHandler)(async (req, res, next) => {
        const wastage = await this.repository.find({
            relations: {
                customer: true,
                factory: true,
                project: true,
                products: true
            },
            order: {
                createdAt: "desc",
            },
        });
        this.success(res, wastage);
    });
    create = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "products", "factoryId");
        let project = null;
        let customer = null;
        if (req.body.customerId) {
            customer = await this.getRepository(entities_1.Customer).findOne({
                where: {
                    id: req.body.customerId,
                },
            });
        }
        if (req.body.projectId) {
            project = await this.getRepository(entities_1.Project).findOne({
                where: {
                    id: req.body.projectId,
                },
            });
        }
        if (req.body.projectId && !project) {
            return this.failed("Project not found", next);
        }
        if (req.body.customerId && !customer) {
            return this.failed("Customer not found", next);
        }
        const factory = await this.getRepository(factory_1.Factory).findOne({
            where: {
                id: req.body.factoryId,
            },
        });
        if (!factory) {
            return this.failed("Factory not found", next);
        }
        const volatileProducts = await Promise.all(req.body.products.map(async ({ id, ...p }) => {
            const product = new wastage_product_1.WastageProduct();
            product.originalId = id;
            product.quantity = p.quantity;
            if (project && project.deliveries) {
                const originalDelivery = await this.getRepository(entities_1.Delivery).findOne({
                    where: {
                        quantity: (0, typeorm_1.MoreThanOrEqual)(p.quantity),
                        project: {
                            id: project.id,
                        },
                    },
                    relations: {
                        products: true,
                    },
                });
                if (!originalDelivery) {
                    throw new Error("Delivery not found");
                }
                const originalProduct = await this.getRepository(volatile_product_1.VolatileProduct).createQueryBuilder('vp')
                    .leftJoinAndSelect("vp.slot", "s")
                    .leftJoinAndSelect("s.project", "p")
                    .andWhere("p.id = :projectId", { projectId: project.id })
                    .getOne();
                if (!product || !originalProduct) {
                    throw new Error("Product not found");
                }
                if (originalDelivery.status === "ON_THE_WAY") {
                    throw new Error("Product not delivered Yet");
                }
                else {
                    originalProduct.deliveredQuantity = Math.max(originalProduct.deliveredQuantity - parseInt(p.quantity), 0);
                }
                await this.getRepository(volatile_product_1.VolatileProduct).save(originalProduct);
            }
            else {
                const factoryProduct = await this.getRepository(FactoryVolatileProduct_1.FactoryVolatileProduct).findOne({
                    where: {
                        factory: {
                            id: factory.id,
                        },
                        originalId: id,
                    },
                });
                if (!factoryProduct) {
                    throw new Error("Factory Product not found");
                }
                factoryProduct.stock -= parseInt(p.quantity);
                const originalProduct = await this.getRepository(entities_1.Product).findOne({
                    where: {
                        id: id,
                    },
                });
                if (!originalProduct) {
                    throw new Error("Product not found");
                }
                originalProduct.stock -= parseInt(p.quantity);
                await this.getRepository(FactoryVolatileProduct_1.FactoryVolatileProduct).save(factoryProduct);
                await this.getRepository(entities_1.Product).save(originalProduct);
            }
            await this.getRepository(wastage_product_1.WastageProduct).save(product);
            return product;
        }));
        if (!volatileProducts.length) {
            throw new Error("Something went wrong");
        }
        const wastage = new wastage_1.Wastage();
        wastage.products = volatileProducts;
        if (customer) {
            wastage.customer = customer;
        }
        if (project) {
            wastage.project = project;
        }
        wastage.factory = factory;
        wastage.quantity = req.body.quantity;
        wastage.reason = req.body?.reason || "Not Specified";
        await this.getRepository(wastage_1.Wastage).save(wastage);
        this.success(res, wastage);
    }, {
        body: {},
    });
}
exports.WastageController = WastageController;
//# sourceMappingURL=wastage.js.map