"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SavingsWithdrawController = void 0;
const entities_1 = require("../entities");
const utils_1 = require("../utils");
const account_1 = require("./account");
const base_1 = require("./base");
class SavingsWithdrawController extends base_1.Controller {
    constructor() {
        super(entities_1.SavingsWithdraw);
    }
    get = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            const qb = this.repository
                .createQueryBuilder("sw")
                .leftJoinAndSelect("sw.customer", "c")
                .select([
                "sw.id",
                "sw.amount",
                "sw.created_at",
                "c.id",
                "c.account_number",
                "c.name",
                "c.mobile",
            ]);
            if (req.query.startDate && req.query.endDate) {
                const isDate = (0, utils_1.isDateValid)(req.query.startDate, req.query.endDate);
                if (!isDate) {
                    throw new utils_1.ErrorHandler("Invalid Date", 404);
                }
                qb.andWhere("DATE(sw.created_at) BETWEEN :startDate AND :endDate", {
                    startDate: req.query.startDate,
                    endDate: req.query.endDate,
                });
            }
            const savingsWithdraw = await qb
                .orderBy("sw.created_at", "DESC")
                .getMany();
            return this.success(res, savingsWithdraw);
        }
        catch (error) {
            return this.failed(error, next);
        }
    }, {
        query: {},
    });
    create = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            this.validateErr(req.body, "amount", "customerId", "accountId");
            const customer = await this.getRepository(entities_1.Customer).findOne({
                where: {
                    id: req.body.customerId,
                },
            });
            if (!customer) {
                return this.failed("Customer not found", next);
            }
            if (customer.savings < parseInt(req.body.amount)) {
                return this.failed("Insufficient savings, with", next);
            }
            customer.savings = Math.max(customer.savings - parseInt(req.body.amount), 0);
            const savingsWithdraw = new entities_1.SavingsWithdraw();
            savingsWithdraw.amount = parseInt(req.body.amount);
            savingsWithdraw.customer = customer;
            await account_1.AccountController.updateBalance(savingsWithdraw.amount, req.body.accountId, entities_1.TransactionType.DEBIT, "Savings Withdraw", "Savings withdraw from savings account", customer);
            await this.repository.save(savingsWithdraw);
            return this.success(res, savingsWithdraw, 201);
        }
        catch (err) {
            return this.failed(err, next);
        }
    }, {
        body: {},
    });
}
exports.SavingsWithdrawController = SavingsWithdrawController;
//# sourceMappingURL=savings-withdraw.js.map