"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReportController = void 0;
const entities_1 = require("../entities");
const orm_config_1 = require("../orm.config");
const utils_1 = require("../utils");
const production_product_1 = require("../entities/production-product");
const delivery_product_1 = require("../entities/delivery-product");
class ReportController {
    getCashFlow = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            const qb = this.getRepository(entities_1.AccountHistory)
                .createQueryBuilder("ah")
                .leftJoinAndSelect("ah.account", "a")
                .leftJoinAndSelect("ah.customer", "c");
            if (req.query.startDate && req.query.endDate) {
                const isValid = (0, utils_1.isDateValid)(req.query.startDate, req.query.endDate);
                if (!isValid) {
                    throw new utils_1.ErrorHandler("Invalid Date", 404);
                }
                qb.andWhere("DATE(ah.createdAt) BETWEEN :startDate AND :endDate", {
                    startDate: req.query.startDate,
                    endDate: req.query.endDate,
                });
            }
            const cashFlow = await qb.orderBy("ah.createdAt", "DESC").getMany();
            this.success(res, cashFlow);
        }
        catch (err) {
            return this.failed(err, next);
        }
    }, {
        query: {
            startDate: "",
            endDate: "",
        },
    });
    getProductionReport = (0, utils_1.requestHandler)(async (req, res, next) => {
        const qb = this.getRepository(production_product_1.ProductionProduct)
            .createQueryBuilder("pp")
            .leftJoinAndSelect("pp.production", "production")
            .leftJoinAndSelect("production.project", "project")
            .leftJoinAndSelect("production.customer", "customer")
            .leftJoinAndSelect("production.factory", "factory");
        if (req.query.startDate && req.query.endDate) {
            const isValid = (0, utils_1.isDateValid)(req.query.startDate, req.query.endDate);
            if (!isValid) {
                throw new utils_1.ErrorHandler("Invalid Date", 404);
            }
            qb.andWhere("DATE(production.createdAt) BETWEEN :startDate AND :endDate", {
                startDate: req.query.startDate,
                endDate: req.query.endDate,
            });
        }
        const data = await qb.getMany();
        // Aggregating the data
        const aggregatedData = {};
        // Loop through the data to aggregate by product name and production date
        data
            .filter((p) => p.production)
            .forEach((item) => {
            const productName = item.name;
            const productionDate = item.production.createdAt.toLocaleDateString(); // Extract the date
            const factoryId = item.production.factory.id;
            const dayShift = item.production.dayShift;
            const nightShift = item.production.nightShift;
            if (!aggregatedData[productName]) {
                aggregatedData[productName] = {};
            }
            if (!aggregatedData[productName][productionDate]) {
                aggregatedData[productName][productionDate] = {};
            }
            if (!aggregatedData[productName][productionDate][factoryId]) {
                aggregatedData[productName][productionDate][factoryId] = {
                    dayShift: 0,
                    nightShift: 0,
                };
            }
            aggregatedData[productName][productionDate][factoryId].dayShift +=
                dayShift;
            aggregatedData[productName][productionDate][factoryId].nightShift +=
                nightShift;
        });
        // Prepare the final table data
        const finalTable = [];
        Object.keys(aggregatedData).forEach((productName) => {
            Object.keys(aggregatedData[productName]).forEach((productionDate) => {
                const factoryData = aggregatedData[productName][productionDate];
                let factory1DayShift = 0;
                let factory1NightShift = 0;
                let factory2DayShift = 0;
                let factory2NightShift = 0;
                let totalDayShift = 0;
                let totalNightShift = 0;
                // Aggregate shifts by factory
                Object.keys(factoryData).forEach((factoryId) => {
                    const shifts = factoryData[parseInt(factoryId)];
                    if (parseInt(factoryId) === 1) {
                        factory1DayShift = shifts.dayShift;
                        factory1NightShift = shifts.nightShift;
                    }
                    else if (parseInt(factoryId) === 2) {
                        factory2DayShift = shifts.dayShift;
                        factory2NightShift = shifts.nightShift;
                    }
                    totalDayShift += shifts.dayShift;
                    totalNightShift += shifts.nightShift;
                });
                // Push the row to finalTable
                finalTable.push({
                    "Product Name": productName,
                    Date: productionDate,
                    "Factory 1 Day Shift": factory1DayShift,
                    "Factory 1 Night Shift": factory1NightShift,
                    "Factory 2 Day Shift": factory2DayShift,
                    "Factory 2 Night Shift": factory2NightShift,
                    "Total Day Shift": totalDayShift,
                    "Total Night Shift": totalNightShift,
                });
            });
        });
        this.success(res, finalTable);
    }, {
        query: {},
    });
    getDeliveryReport = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            const qb = this.getRepository(delivery_product_1.DeliveryProduct)
                .createQueryBuilder("dp")
                .leftJoinAndSelect("dp.delivery", "delivery")
                .leftJoinAndSelect("delivery.factory", "factory");
            if (req.query.startDate && req.query.endDate) {
                const isValid = (0, utils_1.isDateValid)(req.query.startDate, req.query.endDate);
                if (!isValid) {
                    throw new utils_1.ErrorHandler("Invalid Date", 404);
                }
                qb.andWhere("DATE(delivery.createdAt) BETWEEN :startDate AND :endDate", {
                    startDate: req.query.startDate,
                    endDate: req.query.endDate,
                });
            }
            const data = await qb.getMany();
            // Aggregating the data by Product Name and Delivery Date
            const aggregatedData = {};
            // Process each data item and aggregate quantities by product name and delivery date
            data.forEach((item) => {
                const productName = item.name;
                const deliveryDate = item.delivery.createAt.toLocaleDateString().split("T")[0]; // Extract the date from the timestamp
                const factoryId = item.delivery.factory.id;
                const quantity = item.delivery.quantity;
                // Initialize the data structure if it doesn't exist
                if (!aggregatedData[productName]) {
                    aggregatedData[productName] = {};
                }
                if (!aggregatedData[productName][deliveryDate]) {
                    aggregatedData[productName][deliveryDate] = {};
                }
                // Aggregate the quantities by factory
                if (!aggregatedData[productName][deliveryDate][factoryId]) {
                    aggregatedData[productName][deliveryDate][factoryId] = 0;
                }
                aggregatedData[productName][deliveryDate][factoryId] += quantity;
            });
            // Prepare the final table data
            const finalTable = [];
            Object.keys(aggregatedData).forEach((productName) => {
                Object.keys(aggregatedData[productName]).forEach((deliveryDate) => {
                    const factoryData = aggregatedData[productName][deliveryDate];
                    // Initialize factory quantities
                    let factory1Quantity = 0;
                    let factory2Quantity = 0;
                    let totalQuantity = 0;
                    // Sum the quantities for Factory 1 and Factory 2
                    if (factoryData[1])
                        factory1Quantity = factoryData[1];
                    if (factoryData[2])
                        factory2Quantity = factoryData[2];
                    // Calculate the total quantity
                    totalQuantity = factory1Quantity + factory2Quantity;
                    // Push the row to finalTable
                    finalTable.push({
                        "Product Name": productName,
                        "Factory 1": factory1Quantity,
                        "Factory 2": factory2Quantity,
                        Total: totalQuantity,
                    });
                });
            });
            this.success(res, finalTable);
        }
        catch (err) {
            (0, utils_1.ApiError)(err, 404, next);
        }
    }, {
        query: {},
    });
    /**
     * Controller Specific Database Methods
     */
    getRepository = (Entity) => orm_config_1.appDataSource.getRepository(Entity);
    success = (res, data, status = 200, message) => {
        res.status(status).json({
            status,
            success: true,
            message,
            data,
        });
    };
    failed = (err, next, statusCode = 404) => {
        (0, utils_1.ApiError)(err, statusCode, next);
    };
}
exports.ReportController = ReportController;
//# sourceMappingURL=reports.js.map