"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectController = void 0;
const date_fns_tz_1 = require("date-fns-tz");
const entities_1 = require("../entities");
const factory_1 = require("../entities/factory");
const FactoryVolatileProduct_1 = require("../entities/FactoryVolatileProduct");
const volatile_product_1 = require("../entities/volatile-product");
const utils_1 = require("../utils");
const base_1 = require("./base");
const typeorm_1 = require("typeorm");
const slot_1 = require("../entities/slot");
const delivery_product_1 = require("../entities/delivery-product");
class ProjectController extends base_1.Controller {
    constructor() {
        super(entities_1.Project);
    }
    addProject = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "customerId", "products", "totalAmount", "quantity", "projectName");
        const customer = await this.getRepository(entities_1.Customer).findOne({
            where: {
                id: req.body.customerId,
            },
        });
        if (!customer) {
            return this.failed("Customer not found", next);
        }
        const paidAmount = parseFloat(req.body.paidAmount) || 0;
        const totalAmount = parseFloat(req.body.totalAmount) || 0;
        const dueAmount = Math.max(totalAmount - paidAmount, 0);
        const project = new entities_1.Project();
        const volatileProducts = req.body.products.map(async ({ id, ...p }) => {
            const product = new volatile_product_1.VolatileProduct();
            product.originalId = id;
            product.quantity = p.quantity;
            product.deliveredQuantity = 0;
            product.name = p.name;
            product.unitRate = p.unitRate;
            await this.getRepository(volatile_product_1.VolatileProduct).save(product);
            return product;
        });
        const slot = new slot_1.Slot();
        slot.project = project;
        slot.name = "Default Slot";
        slot.products = await Promise.all(volatileProducts);
        try {
            project.customer = customer;
            project.paidAmount = paidAmount;
            project.totalAmount = totalAmount;
            project.dueAmount = dueAmount;
            project.quantity = req.body.quantity;
            project.projectName = req.body.projectName;
            project.address = req.body.address;
            customer.due = Number(customer.due) + dueAmount;
            await Promise.all([
                this.getRepository(entities_1.Customer).save(customer),
                this.repository.save(project),
                this.getRepository(slot_1.Slot).save(slot),
            ]);
        }
        catch (error) {
            return this.failed(error.message, next);
        }
        return this.success(res, project, 201);
    }, {
        body: {},
    });
    getProjects = (0, utils_1.requestHandler)(async (req, res, next) => {
        const projects = await this.repository.find({
            order: {
                createdAt: "desc",
            },
            relations: {
                planProject: {
                    product: true,
                },
                wastages: true,
            },
        });
        this.success(res, projects, 200);
    });
    addDelivery = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            this.validateErr(req.body, "projectId", "products", "quantity", "challanNumber", "trackNumber", "customerId", "factoryId");
            const project = await this.getRepository(entities_1.Project).findOne({
                where: {
                    id: req.body.projectId,
                },
            });
            const factory = await this.getRepository(factory_1.Factory).findOne({
                where: {
                    id: req.body.factoryId,
                },
            });
            const customer = await this.getRepository(entities_1.Customer).findOne({
                where: {
                    id: req.body.customerId,
                },
            });
            if (!customer) {
                return this.failed("Customer not found", next);
            }
            if (!project) {
                return this.failed("Project not found", next);
            }
            if (!factory) {
                return this.failed("Factory not found", next);
            }
            const deliveryProducts = await Promise.all(req.body.products.map(async (testP) => {
                const p = JSON.parse(testP);
                // Fetch the product and production data
                const product = await this.getRepository(volatile_product_1.VolatileProduct).findOne({
                    where: {
                        id: p.id,
                        slot: {
                            project: {
                                id: project.id,
                            },
                        },
                    },
                });
                if (!product) {
                    throw new Error("Something went wrong with product");
                }
                const deliveryProduct = new delivery_product_1.DeliveryProduct();
                deliveryProduct.name = product.name;
                deliveryProduct.originalId = product.originalId;
                deliveryProduct.deliveredQuantity = 0;
                deliveryProduct.pendingQuantiy = p.quantity;
                return await this.getRepository(delivery_product_1.DeliveryProduct).save(deliveryProduct);
            }));
            const products = await Promise.all(req.body.products.map(async (testP) => {
                try {
                    const p = JSON.parse(testP);
                    // Fetch the product and production data
                    const product = await this.getRepository(volatile_product_1.VolatileProduct).findOne({
                        where: {
                            id: p.id,
                            slot: {
                                project: {
                                    id: project.id,
                                },
                            },
                        },
                    });
                    if (!product) {
                        throw new Error("Something went wrong with product");
                    }
                    const originalProduct = await this.getRepository(entities_1.Product).findOne({
                        where: {
                            id: product.originalId,
                        },
                    });
                    if (!originalProduct) {
                        throw new Error("Something wrong with product");
                    }
                    if (!product) {
                        return null; // Prevent further code execution for this product
                    }
                    // Check if requested quantity exceeds product quantity
                    if (p.quantity > product.quantity) {
                        throw new Error("Requested quantity exceeds product quantity");
                    }
                    // Check if delivered quantity exceeds product quantity
                    if (product.deliveredQuantity + p.quantity > product.quantity) {
                        throw new Error("Delivered quantity exceeds product quantity");
                    }
                    const factoryProduct = await this.getRepository(FactoryVolatileProduct_1.FactoryVolatileProduct).findOne({
                        where: {
                            originalId: originalProduct.id,
                            factory: {
                                id: factory.id,
                            },
                        },
                    });
                    if (!factoryProduct) {
                        throw new Error("Factory product not found");
                    }
                    if (p.quantity > factoryProduct.stock) {
                        throw new Error("Stock exceed, Factory may not have enough stock");
                    }
                    // Update product's delivered quantity
                    product.pendingQuantity += p.quantity;
                    originalProduct.stock = Math.max(originalProduct.stock - p.quantity, 0);
                    factoryProduct.stock = Math.max(factoryProduct.stock - p.quantity, 0);
                    await this.getRepository(entities_1.Product).save(originalProduct);
                    await this.getRepository(volatile_product_1.VolatileProduct).save(product);
                    return { ...product, pendingQuantity: p.quantity };
                }
                catch (error) {
                    console.log(error);
                    throw new Error(error);
                }
            }));
            const successfulProducts = products.filter((product) => product !== null);
            if (!successfulProducts) {
                throw new Error("An error occurred while processing the product");
            }
            const delivery = new entities_1.Delivery();
            if (req.file) {
                delivery.url = req.file.filename;
            }
            delivery.quantity = req.body.quantity;
            delivery.products = successfulProducts;
            delivery.project = project;
            delivery.challanNumber = req.body.challanNumber;
            delivery.trackNumber = req.body.trackNumber;
            delivery.customer = customer;
            delivery.factory = factory;
            delivery.deliveryProducts = deliveryProducts;
            if (req.body.deliveryDate) {
                const parsedDate = new Date(req.body.deliveryDate);
                if (!isNaN(parsedDate.getTime())) {
                    delivery.createAt = (0, date_fns_tz_1.toZonedTime)(req.body.deliveryDate, "Asia/Dhaka", {
                        timeZone: "Asia/Dhaka",
                    });
                }
                else {
                    // Handle invalid date if needed
                    throw new Error("Invalid date format");
                }
            }
            await this.getRepository(entities_1.Delivery).save(delivery);
            return this.success(res, delivery, 201);
        }
        catch (error) {
            return this.failed(error.message, next);
        }
    }, {
        body: {},
    });
    getDelivery = (0, utils_1.requestHandler)(async (req, res, next) => {
        let where = {};
        if (req.query.startDate && req.query.endDate) {
            // Convert the endDate to 23:59:59
            let endDate = new Date(req.query.endDate);
            endDate.setHours(23, 59, 59, 999);
            where.createAt = (0, typeorm_1.Between)(req.query.startDate, endDate);
        }
        const delivery = await this.getRepository(entities_1.Delivery).find({
            relations: {
                project: true,
                products: true,
                customer: true,
            },
            where: {
                project: true,
                products: true,
                customer: true,
                ...where,
            },
            order: {
                createAt: "desc",
            },
        });
        this.success(res, delivery);
    }, {
        query: {},
    });
    updateStatus = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "deliveryId");
        const delivery = await this.getRepository(entities_1.Delivery).findOne({
            where: {
                id: req.body.deliveryId,
            },
            relations: {
                products: true,
            },
        });
        if (!delivery) {
            return this.failed("Delivery not found", next);
        }
        delivery.status = "RECIEVED";
        const products = await this.getRepository(volatile_product_1.VolatileProduct).find({
            where: {
                id: (0, typeorm_1.In)(delivery.products.map((p) => p.id)),
            },
        });
        if (!products.length) {
            return this.failed("Products not found", next);
        }
        await Promise.all(products.map(async (p) => {
            const product = await this.getRepository(volatile_product_1.VolatileProduct).findOne({
                where: {
                    id: p.id,
                },
            });
            const deliveryProduct = await this.getRepository(delivery_product_1.DeliveryProduct).findOne({
                where: {
                    originalId: product?.originalId,
                    delivery: {
                        id: delivery.id,
                    },
                },
            });
            if (!product || !deliveryProduct) {
                throw new Error("Something went wrong");
            }
            product.deliveredQuantity += product.pendingQuantity;
            deliveryProduct.deliveredQuantity += product.pendingQuantity;
            product.pendingQuantity = 0;
            await this.getRepository(volatile_product_1.VolatileProduct).save(product);
            await this.getRepository(delivery_product_1.DeliveryProduct).save(deliveryProduct);
        }));
        await this.getRepository(entities_1.Delivery).save(delivery);
        this.success(res, delivery);
    }, {
        body: { deliveryId: 0 },
    });
    uploadDocuments = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "projectId", "name");
        const file = req.file;
        const project = await this.repository.findOne({
            where: {
                id: req.body.projectId,
            },
        });
        if (!project) {
            return this.failed("Project not found", next);
        }
        if (!file) {
            return this.failed("File not found", next);
        }
        const document = new entities_1.Documents();
        document.name = req.body.name;
        document.url = file.filename;
        document.project = project;
        await this.getRepository(entities_1.Documents).save(document);
        this.success(res, document, 201);
    }, {
        body: {},
    });
    uploadDeliveryDocuments = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "deliveryId", "name");
        const file = req.file;
        const project = await this.getRepository(entities_1.Delivery).findOne({
            where: {
                id: req.body.projectId,
            },
        });
        if (!project) {
            return this.failed("Delivery not found", next);
        }
        if (!file) {
            return this.failed("File not found", next);
        }
        const document = new entities_1.Documents();
        document.name = req.body.name;
        document.url = file.filename;
        document.delivery = project;
        await this.getRepository(entities_1.Documents).save(document);
        this.success(res, document, 201);
    }, {
        body: {},
    });
    addOrder = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "projectId", "products", "name");
        const project = await this.repository.findOne({
            where: {
                id: req.body.projectId,
            },
        });
        if (!project) {
            return this.failed("Project not found", next);
        }
        const volatileProducts = await Promise.all(req.body.products.map(async ({ id, ...p }) => {
            const product = new volatile_product_1.VolatileProduct();
            product.originalId = id;
            product.quantity = p.quantity;
            product.deliveredQuantity = 0;
            product.name = p.name;
            product.unitRate = p.unitRate;
            await this.getRepository(volatile_product_1.VolatileProduct).save(product);
            return product;
        }));
        const totalAmount = volatileProducts.reduce((a, b) => a + (parseInt(b.unitRate) || 0) * (parseInt(b.quantity) || 0), 0);
        const quantity = volatileProducts.reduce((a, b) => a + parseInt(b.quantity) || 0, 0);
        const customer = await this.getRepository(entities_1.Customer).findOne({
            where: {
                id: project.customer.id,
            },
        });
        if (!customer) {
            return this.failed("Customer not found", next);
        }
        const slot = new slot_1.Slot();
        slot.project = project;
        slot.products = volatileProducts;
        slot.name = req.body.name;
        project.quantity += quantity;
        project.totalAmount += totalAmount;
        project.dueAmount += totalAmount;
        customer.due += totalAmount;
        await this.getRepository(entities_1.Project).save(project);
        await this.getRepository(entities_1.Customer).save(customer);
        await this.getRepository(slot_1.Slot).save(slot);
        this.success(res, project);
    }, {
        body: {},
    });
    removeOrder = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "projectId", "productId", "slotId");
        const project = await this.repository.findOne({
            where: {
                id: req.body.projectId,
            },
        });
        if (!project) {
            throw new Error("Project not found");
        }
        const customer = await this.getRepository(entities_1.Customer).findOne({
            where: {
                id: project?.customer?.id,
            },
        });
        if (!customer) {
            throw new Error("Customer not found");
        }
        const product = await this.getRepository(volatile_product_1.VolatileProduct).findOne({
            where: {
                id: req.body.productId,
            },
        });
        if (!product) {
            throw new Error("Product not found");
        }
        const slot = await this.getRepository(slot_1.Slot).findOne({
            where: {
                id: req.body.slotId,
            },
        });
        if (!slot) {
            throw new Error("Slot not found");
        }
        slot.products = slot.products.filter((p) => p.id !== product.id);
        await this.getRepository(slot_1.Slot).save(slot);
        const productAmount = product.unitRate * product.quantity;
        project.quantity -= product.quantity;
        project.totalAmount -= productAmount;
        project.dueAmount -= productAmount;
        customer.due -= productAmount;
        await this.getRepository(entities_1.Customer).save(customer);
        await this.getRepository(entities_1.Project).save(project);
        this.success(res, project);
    }, {
        body: {},
    });
}
exports.ProjectController = ProjectController;
//# sourceMappingURL=project.js.map