"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductController = void 0;
const typeorm_1 = require("typeorm");
const entities_1 = require("../entities");
const factory_1 = require("../entities/factory");
const FactoryVolatileProduct_1 = require("../entities/FactoryVolatileProduct");
const opening_quantity_1 = require("../entities/opening-quantity");
const utils_1 = require("../utils");
const base_1 = require("./base");
const volatile_product_1 = require("../entities/volatile-product");
class ProductController extends base_1.Controller {
    constructor() {
        super(entities_1.Product);
    }
    create = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "name", "factories");
        const newProduct = new entities_1.Product();
        Object.assign(newProduct, req.body);
        await Promise.all(req.body.factories.map(async (factory) => {
            const f = await this.getRepository(factory_1.Factory).findOne({
                where: {
                    id: factory.id,
                },
            });
            if (!f) {
                throw new Error("Factory not found");
            }
            const openingQuantity = new opening_quantity_1.OpeningQuantity();
            openingQuantity.factory = f;
            openingQuantity.quantity = Number(factory.openingQuantity);
            newProduct.openingQuantity = newProduct.openingQuantity
                ? [...newProduct.openingQuantity, openingQuantity]
                : [openingQuantity];
            newProduct.stock = newProduct.openingQuantity.reduce((prev, curr) => {
                return prev + Number(curr.quantity);
            }, 0);
            await this.getRepository(factory_1.Factory).save(f);
            await this.getRepository(opening_quantity_1.OpeningQuantity).save(openingQuantity);
        }));
        const product = await this.repository.save(newProduct);
        await Promise.all(req.body.factories.map(async (factory) => {
            const f = await this.getRepository(factory_1.Factory).findOne({
                where: {
                    id: factory.id,
                },
            });
            if (!f) {
                throw new Error("Factory not found");
            }
            const volatile = new FactoryVolatileProduct_1.FactoryVolatileProduct();
            volatile.factory = f;
            volatile.stock = parseInt(factory.openingQuantity);
            volatile.name = product.name;
            volatile.originalId = product.id;
            await this.getRepository(FactoryVolatileProduct_1.FactoryVolatileProduct).save(volatile);
        }));
        const { openingQuantity, ...rest } = product;
        this.success(res, {
            ...rest,
        }, 201);
    }, {
        body: {},
    });
    get = (0, utils_1.requestHandler)(async (req, res, next) => {
        let where = {};
        if (req.query.startDate && req.query.endDate) {
            // Convert the endDate to 23:59:59
            let endDate = new Date(req.query.endDate);
            endDate.setHours(23, 59, 59, 999);
            where.created_at = (0, typeorm_1.Between)(req.query.startDate, endDate);
        }
        const data = await this.repository.find({
            order: {
                created_at: "DESC",
            },
            where,
        });
        return this.success(res, data);
    }, {
        query: {},
    });
    getProductDetails = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.params, "id");
        const product = await this.repository
            .createQueryBuilder("product")
            .leftJoinAndSelect("product.openingQuantity", "openingQuantity")
            .leftJoinAndSelect("openingQuantity.factory", "factory")
            .leftJoinAndSelect("factory.deliveries", "delivery")
            .leftJoinAndSelect("delivery.factory", "deliveryFactory")
            .leftJoinAndSelect("delivery.products", "products")
            .where("product.id=:id", { id: req.params.id })
            .getOne();
        const productions = await this.getRepository(entities_1.Production)
            .createQueryBuilder("pr")
            .leftJoinAndSelect("pr.factory", "factory")
            .leftJoinAndSelect("pr.products", "products")
            .where("products.originalId=:id", { id: req.params.id })
            .getMany();
        const openingQuantity = product?.openingQuantity.map(({ factory, ...o }) => ({
            openingQuantity: {
                ...o,
                factory: factory?.name, // Renaming to 'factory.name'
            },
            type: "openingQuantity", // Tagging the type for later sorting
        }));
        const deliveries = product?.openingQuantity
            .flatMap((p) => p.factory?.deliveries) // Flattening deliveries from factory
            .filter((d) => d.products?.find((p) => p.originalId === product.id)) // Filtering deliveries with the product
            .map(({ products, ...d }) => ({
            ...d,
            factory: d?.factory?.name,
            type: "delivery", // Tagging the type for later sorting
        }));
        if (!productions?.length &&
            !openingQuantity?.length &&
            !deliveries?.length ||
            !product) {
            this.success(res, null, 200, "No data found");
            return;
        }
        this.success(res, {
            productions: productions.map((p) => ({
                ...p,
                quantity: p.products.find((pr) => pr.originalId === product.id)
                    ?.quantity,
            })),
            deliveries,
            openingQuantity,
        });
    }, {
        params: {
            id: 0,
        },
    });
    getVolatileProducts = (0, utils_1.requestHandler)(async (req, res, next) => {
        const data = await this.getRepository(volatile_product_1.VolatileProduct).find();
        return this.success(res, data);
    });
}
exports.ProductController = ProductController;
//# sourceMappingURL=product.js.map