"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccountController = void 0;
const entities_1 = require("../entities");
const orm_config_1 = require("../orm.config");
const utils_1 = require("../utils");
const base_1 = require("./base");
class AccountController extends base_1.Controller {
    constructor() {
        super(entities_1.Account);
    }
    create = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            this.validateErr(req.body, "accountType", "accountName");
            this.isValidEnum(entities_1.AccountType, req.body.accountType);
            const account = new entities_1.Account();
            Object.assign(account, req.body);
            account.balance = req.body.openingBalance.toString();
            const accountHistory = new entities_1.AccountHistory();
            accountHistory.account = account;
            accountHistory.amount = req.body.openingBalance;
            accountHistory.transactionType = entities_1.TransactionType.OPENING_BALANCE;
            await this.getRepository(entities_1.AccountHistory).save(accountHistory);
            return this.success(res, account, 201);
        }
        catch (err) {
            return this.failed(err, next);
        }
    }, {
        body: {},
    });
    get = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            const account = await this.repository
                .createQueryBuilder("a")
                .leftJoinAndSelect("a.accountHistory", "ah")
                .orderBy("a.id", "DESC")
                .orderBy("ah.id", "DESC")
                .getMany();
            return this.success(res, account);
        }
        catch (err) {
            return this.failed(err, next);
        }
    });
    getAccountHistory = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            this.validateErr(req.params, "accountId");
            const qb = this.getRepository(entities_1.AccountHistory)
                .createQueryBuilder("ah")
                .leftJoinAndSelect("ah.account", "a")
                .leftJoinAndSelect("ah.customer", "c")
                .andWhere(`a.id = ${req.params.accountId}`);
            if (req.query.startDate && req.query.endDate) {
                const isValid = (0, utils_1.isDateValid)(req.query.startDate, req.query.endDate);
                if (!isValid) {
                    return this.failed("Invalid Date", next);
                }
                qb.andWhere("DATE(ah.createdAt) BETWEEN :startDate AND :endDate", {
                    startDate: req.query.startDate,
                    endDate: req.query.endDate,
                });
            }
            const account = await qb.orderBy("ah.createdAt", "DESC").getMany();
            return this.success(res, account);
        }
        catch (err) {
            return this.failed(err, next);
        }
    }, {
        query: {},
        params: {},
    });
    transferAmount = (0, utils_1.requestHandler)(async (req, res, next) => {
        this.validateErr(req.body, "senderAccount", "receiverAccount", "amount");
        const cost = parseInt(req.body.cost) || 0;
        const amount = parseInt(req.body.amount) || 0;
        if (req.body.senderAccount === req.body.receiverAccount) {
            return this.failed("Sender and Receiver account can not be same", next);
        }
        const sender = await AccountController.currentAccount(req.body.senderAccount);
        const receiver = await AccountController.currentAccount(req.body.receiverAccount);
        const transfer = new entities_1.AmountTransfer();
        transfer.sender = sender;
        transfer.reciever = receiver;
        transfer.amount = amount;
        transfer.cost = cost;
        await AccountController.updateBalance(amount + cost, req.body.senderAccount, entities_1.TransactionType.DEBIT, "Balance Transfer", "Transfer amount from this account");
        await AccountController.updateBalance(amount, req.body.receiverAccount, entities_1.TransactionType.CREDIT, "Balance Transfer", "Received amount to this account");
        await this.getRepository(entities_1.AmountTransfer).save(transfer);
        return this.success(res, transfer, 201);
    }, {
        body: {},
    });
    getTransferHisoty = (0, utils_1.requestHandler)(async (req, res, next) => {
        try {
            const transfer = await this.getRepository(entities_1.AmountTransfer)
                .createQueryBuilder("at")
                .leftJoinAndSelect("at.sender", "s")
                .leftJoinAndSelect("at.reciever", "r")
                .orderBy("at.id", "DESC")
                .getMany();
            return this.success(res, transfer);
        }
        catch (error) {
            return this.failed(error, next);
        }
    });
    static async currentAccount(accountId) {
        const account = await orm_config_1.appDataSource.getRepository(entities_1.Account).findOne({
            where: {
                id: accountId,
            },
            relations: {
                accountHistory: true,
            },
        });
        if (!account) {
            throw new utils_1.ErrorHandler("Cash Book Not found", 404);
        }
        return account;
    }
    static async updateBalance(amount, accountId, type, remarkType, remark, customer) {
        if (amount === 0 ||
            !amount ||
            amount === undefined ||
            amount === null ||
            Number.isNaN(amount))
            return;
        const account = await AccountController.currentAccount(accountId);
        if (type === "CREDIT") {
            account.balance = Math.round(parseInt(account.balance) + parseInt(amount?.toString())).toString();
        }
        else {
            if (parseInt(account.balance) < parseInt(amount?.toString())) {
                throw new utils_1.ErrorHandler("Insufficient Balance", 404);
            }
            account.balance = Math.round(parseInt(account.balance) - parseInt(amount?.toString())).toString();
        }
        const history = new entities_1.AccountHistory();
        if (customer) {
            history.customer = customer;
        }
        history.account = account;
        history.amount = parseInt(amount?.toString());
        history.transactionType = type;
        history.type = remarkType ?? "";
        history.remark = remark ?? "";
        await orm_config_1.appDataSource.getRepository(entities_1.AccountHistory).save(history);
    }
}
exports.AccountController = AccountController;
//# sourceMappingURL=account.js.map